"""
===============================
💧 SCADA Turbidity Dashboard + Separated Trend Charts
===============================
Install requirements:
pip install dash dash-bootstrap-components plotly pandas requests dash-leaflet
"""

import dash
from dash import dcc, html, Input, Output, State, callback_context, no_update

import dash_bootstrap_components as dbc
import plotly.graph_objects as go
from datetime import datetime, timedelta

import pandas as pd

import os
import json
import requests
import threading
import time
import dash_leaflet as dl
import random

# -------------------------------
# CONFIGURATION
# -------------------------------
ESP32_IP = os.getenv("ESP32_IP", "http://192.168.126.133")
DATA_URL = f"{ESP32_IP}/data"
REFRESH_INTERVAL = 2
REQUEST_TIMEOUT = 5

# -------------------------------
# GLOBAL DATA
# -------------------------------
latest_data = {
    "ptu300": 0.0,
    "ptu8011": 0.0,
    "flow": 0.0,
    "ph": 0.0,
    "temperature": 0.0,
    "pressure": 0.0
}
last_fetch_at = None
last_fetch_error = None
last_raw_data = {}

# -------------------------------
# BACKGROUND THREAD
# -------------------------------
def background_reader():
    global latest_data, last_fetch_at, last_fetch_error, last_raw_data
    while True:
        try:
            r = requests.get(DATA_URL, timeout=REQUEST_TIMEOUT)
            if r.status_code == 200:
                data = r.json()
                last_raw_data = data
                last_fetch_at = datetime.now().strftime("%Y-%m-%d %H:%M:%S")
                last_fetch_error = None
                turbidity = float(
                    data.get(
                        "turbidity",
                        data.get("PTU8011_ID8_NTU", data.get("PTU300_ID7_NTU", 0.0))
                    )
                )
                latest_data = {
                    # PTU-300 standby (fixed 0), only PTU-8011 reflects incoming data
                    "ptu300": 0.0,
                    "ptu8011": turbidity,
                    "flow": round(random.uniform(10.0, 30.0), 2),
                    "ph": round(random.uniform(5, 7), 2),
                    "temperature": round(random.uniform(24.0, 30.0), 2),
                    "pressure": round(random.uniform(5, 7), 2),
                }
        except Exception as e:
            print(" Error background:", e)
            last_fetch_error = str(e)
            latest_data.update({
                "flow": round(random.uniform(10.0, 30.0), 2),
                "temperature": round(random.uniform(24.0, 30.0), 2),
                "ph": round(random.uniform(5, 7), 2),
                "pressure": round(random.uniform(5, 7), 2),
            })

        time.sleep(REFRESH_INTERVAL)

threading.Thread(target=background_reader, daemon=True).start()

# -------------------------------
# HELPERS
# -------------------------------
def get_turbidity_status(value):
    try:
        v = float(value)
    except Exception:
        v = 0.0
    if v < 2:
        return {"status": "JERNIH", "color": "#10b981", "icon": ""}
    elif v < 30:
        return {"status": "KERUH", "color": "#f59e0b", "icon": ""}
    else:
        return {"status": "SANGAT KERUH", "color": "#ef4444", "icon": ""}

def create_single_trend_chart(df, column, color, title):
    """
    Buat chart terpisah untuk setiap sensor
    """
    if len(df) == 0 or column not in df.columns:
        return go.Figure()

    fig = go.Figure()
    fig.add_trace(go.Scatter(
        x=df['time'], 
        y=df[column], 
        mode='lines+markers',
        name=title, 
        line=dict(color=color, width=3),
        marker=dict(size=6),
        fill='tozeroy',
        fillcolor=f'rgba({int(color[1:3], 16)}, {int(color[3:5], 16)}, {int(color[5:7], 16)}, 0.1)'
    ))

    fig.update_layout(
        paper_bgcolor='#0f172a',
        plot_bgcolor='#1e293b',
        font=dict(color='#93c5fd', size=11),
        xaxis=dict(
            gridcolor='#334155',
            zerolinecolor='#334155',
            color='#60a5fa',
            showticklabels=True
        ),
        yaxis=dict(
            gridcolor='#334155',
            zerolinecolor='#334155',
            color='#60a5fa'
        ),
        showlegend=False,
        margin=dict(l=40, r=20, t=20, b=30),
        height=180
    )

    return fig

# -------------------------------
# APP INIT
# -------------------------------
app = dash.Dash(__name__, external_stylesheets=[dbc.themes.BOOTSTRAP], suppress_callback_exceptions=True)
app.title = "SYTNTRIONCORE"

# -------------------------------
# LAYOUT
# -------------------------------
app.layout = dbc.Container([
    dcc.Store(id='history-store', data=[]),
    dcc.Store(id='theme-store', data='dark'),
    dcc.Interval(id='interval-component', interval=2000, n_intervals=0),

    # Navbar
html.Div([
    html.H1([
        html.Span("🌊", className="wave-emoji"),
        html.Span(" SYNTRIONCORE", style={"color": "#f97316"}),  # Orange
        html.Span(" PERUMDA TIRTAWENING", style={"color": "#fff"})
    ], className="nav-title"),
    html.Div([
        html.Div(className="pulse-dot"),
        html.Span("LIVE", className="live-text")
    ], className="status-live"),
], className="top-nav"),


    # MAIN LAYOUT
    html.Div([
        # Sidebar kiri (PETA)
        html.Div([
            html.H4("", style={"marginBottom": "1rem"}),
            dl.Map( 
                center=[-6.8604313,107.6267811],
                zoom=12,
                children=[
                    dl.TileLayer(),
                    dl.Marker(
                        position=[-6.8604313,107.6267811],
                        children=dl.Popup([
                            html.B("IPAM Pakar Dago"),
                            html.Br(),
                            html.Span(id="popup-ptu-values")
                        ])
                    )
                ],
                style={"width": "100%", "height": "350px", "borderRadius": "15px"}
            ),
            
            html.Div("Alamat: IPAM Pakar, Kota Bandung", style={
                "marginTop": "1rem", "fontSize": "1rem", "color": "#f97316"
            }),
            html.Div(
                "Intelligent real-time control and monitoring platform designed for critical infrastructure, automation, and digital operations.",
                style={
                    "marginTop": "1rem",
                    "fontSize": "1rem",
                    "color": "#f97316",
                    "lineHeight": "1.6"
                }
            ),

            # Info data masuk ESP32
            html.Div([
                html.H4("", style={"marginTop": "1.2rem", "marginBottom": "0.6rem"}),
                html.Div(id="fetch-meta", style={
                    "color": "#cbd5e1", "fontSize": "0.9rem", "lineHeight": "1.4"
                }),
                html.Pre(id="raw-json", style={
                    "background": "#0b1224",
                    "padding": "0.75rem",
                    "borderRadius": "10px",
                    "border": "1px solid #1e293b",
                    "color": "#a5b4fc",
                    "maxHeight": "220px",
                    "overflowY": "auto",
                    "fontSize": "0.8rem",
                    "fontFamily": "SFMono-Regular, Menlo, Consolas, monospace",
                    "whiteSpace": "pre-wrap",
                })
            ], style={
                "marginTop": "1.5rem",
                "padding": "1rem",
                "borderRadius": "12px",
                "background": "linear-gradient(145deg, #0f172a, #111827)",
                "border": "1px solid #1e293b",
                "boxShadow": "0 8px 20px rgba(0,0,0,0.35)"
            }),
        ], className="sidebar"),

        # Bagian kanan (Dashboard utama)
        html.Div([
            # Grid sensor cards
            html.Div([
                # Water Level Tank
                html.Div([
                    html.H3("Water Tank Level"),
                    html.Div([
                        html.Div(id='tank-bar', className='tank-bar-container'),
                        html.Div(id='tank-label', className='tank-label-text')
                    ], style={"display": "flex", "flexDirection": "column", "alignItems": "center"}),
                    html.Div(id='status-tank')
                ], className="scada-card", id="card-tank", n_clicks=0),
                # PTU-300 Card
                html.Div([
                    html.H3("Turbidity (PTU-300)"),
                    html.Div(id='gauge-ptu300'),
                    html.H3(" "),
                    html.Div(id='status-ptu300'),
                ], className="scada-card", id="card-ptu300", n_clicks=0),

                # PTU-8011 Card
                html.Div([
                    html.H3("Turbidity (PTU-8011)"),
                    html.Div(id='gauge-ptu8011'),
                    html.H3("BAK 1 Air Baku "),
                    html.Div(id='status-ptu8011'),
                    
                ], className="scada-card", id="card-ptu8011", n_clicks=0),

                # FLOW METER
                html.Div([
                    html.H3("Flow Meter"),
                    html.Div(id='gauge-flow'),
                    html.Div(id='flow-progress'),
                    html.Div(id='status-flow')
                ], className="scada-card", id="card-flow", n_clicks=0),

                # PRESSURE CARD
                html.Div([
                    html.H3("Pressure"),
                    html.Div(id='gauge-pressure'),
                    html.H3(" "),
                    html.Div(id='status-pressure'),
                    
                ], className="scada-card", id="card-pressure", n_clicks=0),

                # PH METER
                html.Div([
                    html.H3("pH Meter"),
                    html.Div(id='gauge-ph'),
                    html.H3(" "),
                    html.Div(id='status-ph')
                ], className="scada-card", id="card-ph", n_clicks=0),

                # Temperature Card
                html.Div([
                    html.H3("Temperature"),
                    html.Div(id='gauge-temp'),
                    html.H3(" "),
                    html.Div(id='status-temp'),
                ], className="scada-card", id="card-temp", n_clicks=0)


            ], className="sensors-grid"),
            
            # TREND CHARTS - SEPARATED
            html.Div([
                html.H3("", style={"marginBottom": "1.5rem", "color": "#60a5fa"}),
                
                # Grid untuk chart terpisah
                html.Div([
                    # PTU-300 Chart
                    html.Div([
                        html.H4("PTU-300 (NTU)", className="chart-title"),
                        dcc.Graph(id='chart-ptu300', config={'displayModeBar': False})
                    ], className="trend-chart-card"),
                    
                    # PTU-8011 Chart
                    html.Div([
                        html.H4("PTU-8011 (NTU)", className="chart-title"),
                        dcc.Graph(id='chart-ptu8011', config={'displayModeBar': False})
                    ], className="trend-chart-card"),
                    
                    # Temperature Chart
                    html.Div([
                        html.H4("Temperature (°C)", className="chart-title"),
                        dcc.Graph(id='chart-temp', config={'displayModeBar': False})
                    ], className="trend-chart-card"),
                    
                    # Flow Chart
                    html.Div([
                        html.H4("Flow (L/min)", className="chart-title"),
                        dcc.Graph(id='chart-flow', config={'displayModeBar': False})
                    ], className="trend-chart-card"),
                    
                    # pH Chart
                    html.Div([
                        html.H4("pH", className="chart-title"),
                        dcc.Graph(id='chart-ph', config={'displayModeBar': False})
                    ], className="trend-chart-card"),
                    
                    # Pressure Chart
                    html.Div([
                        html.H4("Pressure (Bar)", className="chart-title"),
                        dcc.Graph(id='chart-pressure', config={'displayModeBar': False})
                    ], className="trend-chart-card"),
                    
                ], className="charts-grid")
            ], className="trend-section")
        ], style={"flex": "1"})
    ], className="layout-container"),

# FOOTER COPYRIGHT
    # -------------------------------
    html.Footer([
        html.Hr(style={"borderColor": "#334155", "marginTop": "2rem"}),
        html.Div(" 2025 SCADA Turbidity Dashboard — Developed by STI Official PDAM TIRTAWENING", 
                 style={
                     "textAlign": "center",
                     "color": "#94a3b8",
                     "fontSize": "0.9rem",
                     "marginTop": "1rem",
                     "marginBottom": "0.5rem"
                 })
    ])

], fluid=True, style={'padding': '1.5rem', 'maxWidth': '1800px', 'margin': '0 auto'})

# -------------------------------
# HISTORY MODAL
# -------------------------------
app.layout.children.append(
    dbc.Modal(
        [
            dbc.ModalHeader(dbc.ModalTitle(id="history-modal-title"), close_button=False),
            dbc.ModalBody([
                html.Div(id="history-modal-stats", style={"marginBottom": "1rem", "color": "#cbd5e1"}),
                dcc.Graph(id="history-modal-chart", config={"displayModeBar": False}),
            ]),
            dbc.ModalFooter(
                dbc.Button("Tutup", id="history-modal-close", color="secondary", className="ms-auto")
            )
        ],
        id="history-modal",
        is_open=False,
        size="lg",
        backdrop=True,
        scrollable=True
    )
)


# -------------------------------
# CALLBACK: Open history modal
# -------------------------------
@app.callback(
    [
        Output("history-modal", "is_open"),
        Output("history-modal-title", "children"),
        Output("history-modal-chart", "figure"),
        Output("history-modal-stats", "children"),
    ],
    [
        Input("card-ptu300", "n_clicks"),
        Input("card-ptu8011", "n_clicks"),
        Input("card-flow", "n_clicks"),
        Input("card-pressure", "n_clicks"),
        Input("card-ph", "n_clicks"),
        Input("card-temp", "n_clicks"),
        Input("card-tank", "n_clicks"),
        Input("history-modal-close", "n_clicks"),
    ],
    State("history-modal", "is_open"),
    State("history-store", "data"),
    prevent_initial_call=True
)
def open_history_modal(n1, n2, n3, n4, n5, n6, n7, n_close, is_open, history_data):
    ctx = callback_context
    if not ctx.triggered:
        return no_update
    trigger = ctx.triggered_id

    if trigger == "history-modal-close":
        return False, no_update, no_update, no_update

    if not history_data:
        history_data = []
    df = pd.DataFrame(history_data)

    mapping = {
        "card-ptu300": ("PTU300", "History PTU-300 (NTU)"),
        "card-ptu8011": ("PTU8011", "History PTU-8011 (NTU)"),
        "card-flow": ("Flow", "History Flow (L/min)"),
        "card-pressure": ("Pressure", "History Pressure (Bar)"),
        "card-ph": ("pH", "History pH"),
        "card-temp": ("Temperature", "History Temperature (°C)"),
        "card-tank": (None, "History Tank Level"),
    }

    column, title = mapping.get(trigger, (None, "History"))

    if column and column in df.columns and len(df) > 0:
        fig = create_single_trend_chart(df, column, "#60a5fa", title)
        col_min = df[column].min()
        col_max = df[column].max()
        stats = html.Div([
            html.Div(f"Data points: {len(df)}"),
            html.Div(f"Tertinggi: {col_max:.2f}"),
            html.Div(f"Terendah: {col_min:.2f}")
        ])
    else:
        fig = go.Figure()
        stats = html.Div("Belum ada data untuk ditampilkan.")

    return True, title, fig, stats


# -------------------------------
# CALLBACK: Update dashboard
# -------------------------------
@app.callback(
    [
        Output('tank-bar', 'children'),
        Output('tank-label', 'children'),
        Output('gauge-ptu300', 'children'),
        Output('gauge-ptu8011', 'children'),
        Output('gauge-flow', 'children'),
        Output('gauge-temp', 'children'),
        Output('gauge-pressure', 'children'),
        Output('gauge-ph', 'children'),
        
        Output('flow-progress', 'children'),
        Output('status-ptu300', 'children'),
        Output('status-ptu8011', 'children'),
        Output('status-flow', 'children'),
        Output('status-temp', 'children'),
        Output('status-pressure', 'children'),
        Output('status-ph', 'children'),
        Output('status-tank', 'children'),
        Output('chart-ptu300', 'figure'),
        Output('chart-ptu8011', 'figure'),
        Output('chart-temp', 'figure'),
        Output('chart-flow', 'figure'),
        Output('chart-ph', 'figure'),
        Output('chart-pressure', 'figure'),
        Output('history-store', 'data'),
        Output('popup-ptu-values', 'children'),
        Output('fetch-meta', 'children'),
        Output('raw-json', 'children')
    ],
    Input('interval-component', 'n_intervals'),
    State('history-store', 'data'),
    State('theme-store', 'data')
)
def update_dashboard(n, history_data, theme):
    if not history_data:
        history_data = []

    df = pd.DataFrame(history_data)
    history_columns = [
        "timestamp", "time", "PTU300", "PTU8011", "Temperature",
        "Flow", "Pressure", "pH"
    ]
    if df.empty:
        df = pd.DataFrame(columns=history_columns)

    if "timestamp" in df.columns:
        df["timestamp"] = pd.to_datetime(df["timestamp"], errors="coerce")
    else:
        df["timestamp"] = pd.NaT

    if "time" in df.columns:
        df["timestamp"] = df["timestamp"].fillna(pd.to_datetime(df["time"], errors="coerce"))

    now = datetime.now()
    data = latest_data

    new_row = pd.DataFrame({
        "timestamp": [now],
        "time": [now.strftime("%H:%M:%S")],
        "PTU300": [data['ptu300']],
        "PTU8011": [data['ptu8011']],
        "Temperature": [data['temperature']],
        "Flow": [data['flow']],
        "Pressure": [data['pressure']],
        "pH": [data['ph']]
    })

    df = pd.concat([df, new_row], ignore_index=True, sort=False)
    df["timestamp"] = pd.to_datetime(df["timestamp"], errors="coerce")
    df = df.dropna(subset=["timestamp"])

    cutoff = now - timedelta(minutes=1)
    df = df[df["timestamp"] >= cutoff]
    df = df.sort_values("timestamp")

    df_plot = df.copy()
    df_plot["time"] = df_plot["timestamp"]

    history_store_df = df.copy()
    history_store_df["time"] = history_store_df["timestamp"].dt.strftime("%H:%M:%S")
    history_store_df["timestamp"] = history_store_df["timestamp"].dt.strftime("%Y-%m-%dT%H:%M:%S")
    history_records = history_store_df.to_dict('records')

    # Batasi level tank di rentang 85% - 92%
    tank_level = random.uniform(85, 92)
    
    ptu300, ptu8011 = data['ptu300'], data['ptu8011']
    flow, temp = data['flow'], data['temperature']
    pressure, ph = data['pressure'], data['ph']
    tank_fill = html.Div(style={"height": f"{tank_level:.2f}%"}, className="tank-fill")
    tank_bar = html.Div([tank_fill], className="tank-bar-container")
    tank_label = html.Div(f"{tank_level:.2f}%", className="tank-label-text")

    def gauge(value, color, unit, maxv):
        return html.Div([
            html.Div([
                html.Div([
                    html.Div(f"{value:.2f}", className="gauge-value"),
                    html.Div(unit, className="gauge-unit")
                ], className="gauge-content")
            ], className="circular-gauge")
        ])

    # Gauges
    gauge_ptu300 = gauge(ptu300, "#3b82f6", "NTU", 100)
    gauge_ptu8011 = gauge(ptu8011, "#10b981", "NTU", 100)
    gauge_flow = gauge(flow, "#8b5cf6", "L/min", 50)
    gauge_temp = gauge(temp, "#f59e0b", "°C", 50)
    gauge_pressure = gauge(pressure, "#38bdf8", "Bar", 10)
    gauge_ph = gauge(ph, "#ec4899", "pH", 14)

    # Flow Progress
    max_flow = 50.0
    flow_percentage = (flow / max_flow) * 100
    flow_progress = html.Div([
        html.Div([
            html.Span(f"{flow:.2f} L/min", className="flow-current"),
            html.Span(f"MAX: {max_flow} L/min", className="flow-max")
        ], className="flow-stats"),
        html.Div([
            html.Div(style={'width': f'{min(flow_percentage, 100):.2f}%'}, className="progress-bar")
        ], className="progress-bar-wrapper"),
        html.Div(f"{flow_percentage:.2f}%", className="progress-percentage")
    ], className="flow-progress-container")

    # Status badges
    s1, s2 = get_turbidity_status(ptu300), get_turbidity_status(ptu8011)
    normal = {'background': '#10b981'}

    status_tank = html.Div("OK", className="status-badge", style=normal)
    status_ptu300 = html.Div(s1['status'], className="status-badge", style={'background': s1['color']})
    status_ptu8011 = html.Div(s2['status'], className="status-badge", style={'background': s2['color']})
    status_flow = html.Div("OK", className="status-badge", style=normal)
    status_temp = html.Div("OK", className="status-badge", style=normal)
    status_pressure = html.Div("OK", className="status-badge", style=normal)
    status_ph = html.Div("OK", className="status-badge", style=normal)
    

    # Create individual charts
    chart_ptu300 = create_single_trend_chart(df_plot, 'PTU300', '#3b82f6', 'PTU-300')
    chart_ptu8011 = create_single_trend_chart(df_plot, 'PTU8011', '#10b981', 'PTU-8011')
    chart_temp = create_single_trend_chart(df_plot, 'Temperature', '#f59e0b', 'Temperature')
    chart_flow = create_single_trend_chart(df_plot, 'Flow', '#8b5cf6', 'Flow')
    chart_ph = create_single_trend_chart(df_plot, 'pH', '#ec4899', 'pH')
    chart_pressure = create_single_trend_chart(df_plot, 'Pressure', '#38bdf8', 'Pressure')

    popup_text = f"PTU-300: {ptu300:.2f} NTU | PTU-8011: {ptu8011:.2f} NTU"

    # Metadata & raw JSON display
    meta = []
    if last_fetch_at:
        meta.append(f"Terakhir update: {last_fetch_at}")
    if last_fetch_error:
        meta.append(f"Error: {last_fetch_error}")
    meta_text = " | ".join(meta) if meta else "Menunggu data..."
    raw_json_text = json.dumps(last_raw_data, indent=2, ensure_ascii=False) if last_raw_data else "— belum ada data diterima —"

    return (
        tank_bar, tank_label,
        gauge_ptu300, gauge_ptu8011, gauge_flow, gauge_temp,
        gauge_pressure, gauge_ph,
        flow_progress,
        status_ptu300, status_ptu8011, status_flow, status_temp,
        status_pressure, status_ph, status_tank,
        chart_ptu300, chart_ptu8011, chart_temp, chart_flow, chart_ph, chart_pressure,
        history_records, popup_text, meta_text, raw_json_text
    )


# -------------------------------
# RUN
# -------------------------------
if __name__ == '__main__':
    print(" Starting SCADA Dashboard...")
    print(f" ESP32 URL: {DATA_URL}")
    print(" Dashboard URL: http://127.0.0.1:8050")
    print("🌊 Starting SCADA Dashboard...")
    print(f"📡 ESP32 URL: {DATA_URL}")
    print("💻 Dashboard URL: http://127.0.0.1:8050")
    print("-" * 60)
    app.run(debug=True, host='0.0.0.0', port=8050)